package editor;

import game.IntRect;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.font.FontRenderContext;
import java.awt.geom.Rectangle2D;
import java.awt.image.ImageObserver;
import java.io.BufferedWriter;

public class DoorSourceScreenObject extends ScreenObject implements CloneableScreenObject {
	
	private DoorSourcePaletteObject paletteObject;
	private IntRect clickRect;
	private boolean isSelected;
	
	public DoorSourceScreenObject(int x, int y, DoorSourcePaletteObject thePaletteObject) {
		
		paletteObject = thePaletteObject;
		
		int width = paletteObject.getWidth();
		int height = paletteObject.getHeight();
		
		clickRect = new IntRect(x - width/2, y - height/2, width, height);
		
		
	}
	
	
	@Override
	public void draw(Graphics g) {
		
		paletteObject.getImage().drawObjectFromUpperLeftCorner(g, clickRect.getMinX(), clickRect.getMinY());
		
		Color origCol = g.getColor();
		if(paletteObject.shouldShowCaption()) {
			FontRenderContext frc = g.getFontMetrics().getFontRenderContext();
			String dispStr = "to: " + paletteObject.getDestinationRoom().getStaticBlockFactory().getRoomName();
			Rectangle2D strBounds = g.getFont().getStringBounds(dispStr, frc);
			g.setColor(Color.white);
			g.fillRect(clickRect.getMinX(), clickRect.getMaxY() + 2, 4 + (int) strBounds.getWidth(), 4 + (int) strBounds.getHeight());
			g.setColor(Color.black);
			g.drawString(dispStr, clickRect.getMinX() + 2,  clickRect.getMaxY() + 18);
			
		}
		
		if(paletteObject.getCaption() != null && paletteObject.getCaptionColor() != null) {
			g.setColor(paletteObject.getCaptionColor());
			
			Font origFont = g.getFont();
			Font newFont = new Font(origFont.getName(), Font.BOLD, 18);
			g.setFont(newFont);
			g.drawString(paletteObject.getCaption(), clickRect.getMinX() + 4, clickRect.getMaxY() - 10);
			g.setFont(origFont);
		}
		
		if(isSelected)
			drawSelectionOverObject(g);
		
		g.setColor(origCol);
	}

	@Override
	public IntRect getClickRect() {
		return clickRect;
	}

	@Override
	public void setClickRect(IntRect newRect) {
		clickRect = newRect;
	}

	@Override
	public void setSelected(boolean is) {
		isSelected = is;
	}

	@Override
	public boolean isSelected() {
		return isSelected;
	}
	
	
	public String writeCommentDoorOutToFile(BufferedWriter out)throws Exception {
		return writeDoorOutToFile(out, "comment_");
	}
	
	public String writeDoorOutToFile(BufferedWriter out)throws Exception {
		return writeDoorOutToFile(out, "");
	}
	
	public String writeDoorOutToFile(BufferedWriter out, String prefix) throws Exception {
		
		out.write(prefix);
		
		if(paletteObject.getDestination().getDestionationPoint() == null) {
			
			
			out.write("portaNoCompleta");
			out.newLine();
			
			Point centerPoint = clickRect.getCenter();
			
			out.write("x:");
			out.write(Integer.toString(centerPoint.x));
			out.newLine();
			
			out.write("y:");
			out.write(Integer.toString(centerPoint.y));
			out.newLine();
			
			
			out.write("graphic:");
			out.write(paletteObject.getImage().getFileName());
			out.write("*");
			out.write(CustomAnimationPanel.convertColorToString(paletteObject.getImage().getTransparentColor()));
			out.newLine();
			
			out.write("destinationRoom:");
			out.write(paletteObject.getDestinationRoom().getStaticBlockFactory().getRoomName());
			out.newLine();
			
			out.write("/portaNoCompleta");
			out.newLine();
			
			out.newLine();
			
			
			return "Error:  a door in room " + paletteObject.getSourceRoom().getStaticBlockFactory().getRoomName() + " is supposed to lead to room " + paletteObject.getDestinationRoom().getStaticBlockFactory().getRoomName() + ", but has no destination point set.";
		}
		
		out.write("door");
		out.newLine();
		
		Point centerPoint = clickRect.getCenter();
		
		out.write("x:");
		out.write(Integer.toString(centerPoint.x));
		out.newLine();
		
		out.write("y:");
		out.write(Integer.toString(centerPoint.y));
		out.newLine();
		
		
		out.write("graphic:");
		out.write(paletteObject.getImage().getFileName());
		out.write("*");
		out.write(CustomAnimationPanel.convertColorToString(paletteObject.getImage().getTransparentColor()));
		out.newLine();
		
		out.write("destinationRoom:");
		out.write(paletteObject.getDestinationRoom().getStaticBlockFactory().getRoomName());
		out.newLine();
		
		Point targetPoint = paletteObject.getDestination().getDestionationPoint().getClickRect().getCenter();
		
		out.write("destinationX:");
		out.write(Integer.toString(targetPoint.x));
		out.newLine();
		
		out.write("destinationY:");
		out.write(Integer.toString(targetPoint.y));
		out.newLine();
		
		out.write("/door");
		out.newLine();
		
		out.newLine();
		
		return null;
	}
	
	
	public DoorSourceScreenObject clone(ImageObserver io) {
		
		Point centerPoint = clickRect.getCenter();
		
		DoorSourceScreenObject returnValue = new DoorSourceScreenObject(centerPoint.x, centerPoint.y, this.paletteObject);
		return returnValue;
		
	}
	
	@Override
	public int getLayer() {
		
		return 5;
	}
	
	
	public boolean isConnected() {
		
		if(paletteObject != null) {
			DoorDestinationPaletteObject destPO = paletteObject.getDestination();
			if(destPO != null && destPO.getDestionationPoint() != null)
				return true;
		}
		
		
		return false;
		
	}
	
	public RoomEditorPanel getTargetRoomPanel() {
		
		if(paletteObject == null)
			return null;
		
		return paletteObject.getDestinationRoom();
		
	}

}
